// Copyright 2018-2024 the Deno authors. All rights reserved. MIT license.
// Copyright 2020 Keith Cirkel. All rights reserved. MIT license.
// Copyright 2023 Skye "MierenManz". All rights reserved. MIT license.
/**
 * Functions for encoding typed integers in array buffers.
 *
 * @module
 */ // This implementation is a port of https://deno.land/x/varint@v2.0.0 by @keithamus
// This module is browser compatible.
export const MaxUInt64 = 18446744073709551615n;
export const MaxVarIntLen64 = 10;
export const MaxVarIntLen32 = 5;
const MSB = 0x80;
const REST = 0x7f;
const SHIFT = 7;
const MSBN = 0x80n;
const SHIFTN = 7n;
// ArrayBuffer and TypedArray's for "pointer casting"
const AB = new ArrayBuffer(8);
const U32_VIEW = new Uint32Array(AB);
const U64_VIEW = new BigUint64Array(AB);
/**
 * Given a non empty `buf`, starting at `offset` (default: 0), begin decoding bytes as
 * VarInt encoded bytes, for a maximum of 10 bytes (offset + 10). The returned
 * tuple is of the decoded varint 32-bit number, and the new offset with which
 * to continue decoding other data.
 *
 * If a `bigint` in return is undesired, the `decode32` function will return a
 * `number`, but this should only be used in cases where the varint is
 * _assured_ to be 32-bits. If in doubt, use `decode()`.
 *
 * To know how many bytes the VarInt took to encode, simply negate `offset`
 * from the returned new `offset`.
 */ export function decode(buf, offset = 0) {
  // Clear the last result from the Two's complement view
  U64_VIEW[0] = 0n;
  // Setup the initiat state of the function
  let intermediate = 0;
  let position = 0;
  let i = offset;
  // If the buffer is empty Throw
  if (buf.length === 0) throw new RangeError("Cannot read empty buffer");
  let byte;
  do {
    // Get a single byte from the buffer
    byte = buf[i];
    // 1. Take the lower 7 bits of the byte.
    // 2. Shift the bits into the correct position.
    // 3. Bitwise OR it with the intermediate value
    // QUIRK: in the 5th (and 10th) iteration of this loop it will overflow on the shift.
    // This causes only the lower 4 bits to be shifted into place and removing the upper 3 bits
    intermediate |= (byte & 0b01111111) << position;
    // If position is 28
    // it means that this iteration needs to be written the the two's complement view
    // This only happens once due to the `-4` in this branch
    if (position === 28) {
      // Write to the view
      U32_VIEW[0] = intermediate;
      // set `intermediate` to the remaining 3 bits
      // We only want the remaining three bits because the other 4 have been "consumed" on line 21
      intermediate = (byte & 0b01110000) >>> 4;
      // set `position` to -4 because later 7 will be added, making it 3
      position = -4;
    }
    // Increment the shift position by 7
    position += 7;
    // Increment the iterator by 1
    i++;
  // Keep going while there is a continuation bit
  }while ((byte & 0b10000000) === 0b10000000)
  // subtract the initial offset from `i` to get the bytes read
  const nRead = i - offset;
  // If 10 bytes have been read and intermediate has overflown
  // it means that the varint is malformed
  // If 11 bytes have been read it means that the varint is malformed
  // If `i` is bigger than the buffer it means we overread the buffer and the varint is malformed
  if (nRead === 10 && intermediate > -1 || nRead === 11 || i > buf.length) {
    throw new RangeError("malformed or overflow varint");
  }
  // Write the intermediate value to the "empty" slot
  // if the first slot is taken. Take the second slot
  U32_VIEW[Number(nRead > 4)] = intermediate;
  return [
    U64_VIEW[0],
    i
  ];
}
/**
 * Given a `buf`, starting at `offset` (default: 0), begin decoding bytes as
 * VarInt encoded bytes, for a maximum of 5 bytes (offset + 5). The returned
 * tuple is of the decoded varint 32-bit number, and the new offset with which
 * to continue decoding other data.
 *
 * VarInts are _not 32-bit by default_ so this should only be used in cases
 * where the varint is _assured_ to be 32-bits. If in doubt, use `decode()`.
 *
 * To know how many bytes the VarInt took to encode, simply negate `offset`
 * from the returned new `offset`.
 */ export function decode32(buf, offset = 0) {
  for(let i = offset, len = Math.min(buf.length, offset + MaxVarIntLen32), shift = 0, decoded = 0; i <= len; i += 1, shift += SHIFT){
    const byte = buf[i];
    decoded += (byte & REST) * Math.pow(2, shift);
    if (!(byte & MSB)) return [
      decoded,
      i + 1
    ];
  }
  throw new RangeError("malformed or overflow varint");
}
/**
 * Takes unsigned number `num` and converts it into a VarInt encoded
 * `Uint8Array`, returning a tuple consisting of a `Uint8Array` slice of the
 * encoded VarInt, and an offset where the VarInt encoded bytes end within the
 * `Uint8Array`.
 *
 * If `buf` is not given then a Uint8Array will be created.
 * `offset` defaults to `0`.
 *
 * If passed `buf` then that will be written into, starting at `offset`. The
 * resulting returned `Uint8Array` will be a slice of `buf`. The resulting
 * returned number is effectively `offset + bytesWritten`.
 */ export function encode(num, buf = new Uint8Array(MaxVarIntLen64), offset = 0) {
  num = BigInt(num);
  if (num < 0n) throw new RangeError("signed input given");
  for(let i = offset, len = Math.min(buf.length, MaxVarIntLen64); i <= len; i += 1){
    if (num < MSBN) {
      buf[i] = Number(num);
      i += 1;
      return [
        buf.slice(offset, i),
        i
      ];
    }
    buf[i] = Number(num & 0xFFn | MSBN);
    num >>= SHIFTN;
  }
  throw new RangeError(`${num} overflows uint64`);
}
//# sourceMappingURL=data:application/json;base64,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