// Copyright 2018-2024 the Deno authors. All rights reserved. MIT license.
// This module is browser compatible.
import { assert } from "../assert/assert.ts";
import { exponentialBackoffWithJitter } from "./_util.ts";
/**
 * Error thrown in {@linkcode retry} once the maximum number of failed attempts
 * has been reached.
 */ export class RetryError extends Error {
  /**
   * Constructs a new {@linkcode RetryError} instance.
   *
   * @example
   * ```
   * import { RetryError } from "https://deno.land/std@$STD_VERSION/async/retry.ts";
   *
   * throw new RetryError({ foo: "bar" }, 3);
   * ```
   */ constructor(cause, attempts){
    super(`Retrying exceeded the maxAttempts (${attempts}).`);
    this.name = "RetryError";
    this.cause = cause;
  }
}
const defaultRetryOptions = {
  multiplier: 2,
  maxTimeout: 60000,
  maxAttempts: 5,
  minTimeout: 1000,
  jitter: 1
};
/**
 * Calls the given (possibly asynchronous) function up to `maxAttempts` times.
 * Retries as long as the given function throws. If the attempts are exhausted,
 * throws a {@linkcode RetryError} with `cause` set to the inner exception.
 *
 * The backoff is calculated by multiplying `minTimeout` with `multiplier` to the power of the current attempt counter (starting at 0 up to `maxAttempts - 1`). It is capped at `maxTimeout` however.
 * How long the actual delay is, depends on `jitter`.
 *
 * When `jitter` is the default value of `1`, waits between two attempts for a
 * randomized amount between 0 and the backoff time. With the default options
 * the maximal delay will be `15s = 1s + 2s + 4s + 8s`. If all five attempts
 * are exhausted the mean delay will be `9.5s = ½(4s + 15s)`.
 *
 * When `jitter` is `0`, waits the full backoff time.
 *
 * @example
 * ```ts
 * import { retry } from "https://deno.land/std@$STD_VERSION/async/retry.ts";
 * const req = async () => {
 *  // some function that throws sometimes
 * };
 *
 * // Below resolves to the first non-error result of `req`
 * const retryPromise = await retry(req, {
 *  multiplier: 2,
 *  maxTimeout: 60000,
 *  maxAttempts: 5,
 *  minTimeout: 100,
 *  jitter: 1,
 * });
 * ```
 *
 * @example
 * ```ts
 * import { retry } from "https://deno.land/std@$STD_VERSION/async/retry.ts";
 * const req = async () => {
 *  // some function that throws sometimes
 * };
 *
 * // Make sure we wait at least 1 minute, but at most 2 minutes
 * const retryPromise = await retry(req, {
 *  multiplier: 2.34,
 *  maxTimeout: 80000,
 *  maxAttempts: 7,
 *  minTimeout: 1000,
 *  jitter: 0.5,
 * });
 * ```
 */ export async function retry(fn, opts) {
  const options = {
    ...defaultRetryOptions,
    ...opts
  };
  assert(options.maxTimeout >= 0, "maxTimeout is less than 0");
  assert(options.minTimeout <= options.maxTimeout, "minTimeout is greater than maxTimeout");
  assert(options.jitter <= 1, "jitter is greater than 1");
  let attempt = 0;
  while(true){
    try {
      return await fn();
    } catch (error) {
      if (attempt + 1 >= options.maxAttempts) {
        throw new RetryError(error, options.maxAttempts);
      }
      const timeout = exponentialBackoffWithJitter(options.maxTimeout, options.minTimeout, attempt, options.multiplier, options.jitter);
      await new Promise((r)=>setTimeout(r, timeout));
    }
    attempt++;
  }
}
//# sourceMappingURL=data:application/json;base64,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