// Copyright 2018-2024 the Deno authors. All rights reserved. MIT license.
// This module is browser compatible.
/**
 * Command line arguments parser based on
 * {@link https://github.com/minimistjs/minimist | minimist}.
 *
 * This module is browser compatible.
 *
 * @example
 * ```ts
 * import { parseArgs } from "https://deno.land/std@$STD_VERSION/cli/parse_args.ts";
 *
 * console.dir(parseArgs(Deno.args));
 * ```
 *
 * @module
 */ import { assert } from "../assert/assert.ts";
function isNumber(x) {
  if (typeof x === "number") return true;
  if (/^0x[0-9a-f]+$/i.test(String(x))) return true;
  return /^[-+]?(?:\d+(?:\.\d*)?|\.\d+)(e[-+]?\d+)?$/.test(String(x));
}
function setNested(object, keys, value, collect = false) {
  keys.slice(0, -1).forEach((key)=>{
    object[key] ??= {};
    object = object[key];
  });
  const key = keys[keys.length - 1];
  if (collect) {
    const v = object[key];
    if (Array.isArray(v)) {
      v.push(value);
      return;
    }
    value = v ? [
      v,
      value
    ] : [
      value
    ];
  }
  object[key] = value;
}
function hasNested(object, keys) {
  keys = [
    ...keys
  ];
  const lastKey = keys.pop();
  if (!lastKey) return false;
  for (const key of keys){
    if (!object[key]) return false;
    object = object[key];
  }
  return Object.hasOwn(object, lastKey);
}
function aliasIsBoolean(aliasMap, booleanSet, key) {
  const set = aliasMap.get(key);
  if (set === undefined) return false;
  for (const alias of set)if (booleanSet.has(alias)) return true;
  return false;
}
function isBooleanString(value) {
  return value === "true" || value === "false";
}
function parseBooleanString(value) {
  return value !== "false";
}
const FLAG_REGEXP = /^(?:-(?:(?<doubleDash>-)(?<negated>no-)?)?)(?<key>.+?)(?:=(?<value>.+?))?$/s;
/**
 * Take a set of command line arguments, optionally with a set of options, and
 * return an object representing the flags found in the passed arguments.
 *
 * By default, any arguments starting with `-` or `--` are considered boolean
 * flags. If the argument name is followed by an equal sign (`=`) it is
 * considered a key-value pair. Any arguments which could not be parsed are
 * available in the `_` property of the returned object.
 *
 * By default, the flags module tries to determine the type of all arguments
 * automatically and the return type of the `parseArgs` method will have an index
 * signature with `any` as value (`{ [x: string]: any }`).
 *
 * If the `string`, `boolean` or `collect` option is set, the return value of
 * the `parseArgs` method will be fully typed and the index signature of the return
 * type will change to `{ [x: string]: unknown }`.
 *
 * Any arguments after `'--'` will not be parsed and will end up in `parsedArgs._`.
 *
 * Numeric-looking arguments will be returned as numbers unless `options.string`
 * or `options.boolean` is set for that argument name.
 *
 * @example
 * ```ts
 * import { parseArgs } from "https://deno.land/std@$STD_VERSION/cli/parse_args.ts";
 * const parsedArgs = parseArgs(Deno.args);
 * ```
 *
 * @example
 * ```ts
 * import { parseArgs } from "https://deno.land/std@$STD_VERSION/cli/parse_args.ts";
 * const parsedArgs = parseArgs(["--foo", "--bar=baz", "./quux.txt"]);
 * // parsedArgs: { foo: true, bar: "baz", _: ["./quux.txt"] }
 * ```
 */ export function parseArgs(args, { "--": doubleDash = false, alias = {}, boolean = false, default: defaults = {}, stopEarly = false, string = [], collect = [], negatable = [], unknown: unknownFn = (i)=>i } = {}) {
  const aliasMap = new Map();
  const booleanSet = new Set();
  const stringSet = new Set();
  const collectSet = new Set();
  const negatableSet = new Set();
  let allBools = false;
  if (alias) {
    for(const key in alias){
      const val = alias[key];
      assert(val !== undefined);
      const aliases = Array.isArray(val) ? val : [
        val
      ];
      aliasMap.set(key, new Set(aliases));
      const set = new Set([
        key,
        ...aliases
      ]);
      aliases.forEach((alias)=>aliasMap.set(alias, set));
    }
  }
  if (boolean) {
    if (typeof boolean === "boolean") {
      allBools = boolean;
    } else {
      const booleanArgs = Array.isArray(boolean) ? boolean : [
        boolean
      ];
      for (const key of booleanArgs.filter(Boolean)){
        booleanSet.add(key);
        aliasMap.get(key)?.forEach((al)=>{
          booleanSet.add(al);
        });
      }
    }
  }
  if (string) {
    const stringArgs = Array.isArray(string) ? string : [
      string
    ];
    for (const key of stringArgs.filter(Boolean)){
      stringSet.add(key);
      aliasMap.get(key)?.forEach((al)=>stringSet.add(al));
    }
  }
  if (collect) {
    const collectArgs = Array.isArray(collect) ? collect : [
      collect
    ];
    for (const key of collectArgs.filter(Boolean)){
      collectSet.add(key);
      aliasMap.get(key)?.forEach((al)=>collectSet.add(al));
    }
  }
  if (negatable) {
    const negatableArgs = Array.isArray(negatable) ? negatable : [
      negatable
    ];
    for (const key of negatableArgs.filter(Boolean)){
      negatableSet.add(key);
      aliasMap.get(key)?.forEach((alias)=>negatableSet.add(alias));
    }
  }
  const argv = {
    _: []
  };
  function setArgument(key, value, arg, collect) {
    if (!booleanSet.has(key) && !stringSet.has(key) && !aliasMap.has(key) && !(allBools && /^--[^=]+$/.test(arg)) && unknownFn?.(arg, key, value) === false) {
      return;
    }
    if (typeof value === "string" && !stringSet.has(key)) {
      value = isNumber(value) ? Number(value) : value;
    }
    const collectable = collect && collectSet.has(key);
    setNested(argv, key.split("."), value, collectable);
    aliasMap.get(key)?.forEach((key)=>setNested(argv, key.split("."), value, collectable));
  }
  let notFlags = [];
  // all args after "--" are not parsed
  const index = args.indexOf("--");
  if (index !== -1) {
    notFlags = args.slice(index + 1);
    args = args.slice(0, index);
  }
  for(let i = 0; i < args.length; i++){
    const arg = args[i];
    const groups = arg.match(FLAG_REGEXP)?.groups;
    if (groups) {
      const { doubleDash, negated } = groups;
      let key = groups.key;
      let value = groups.value;
      if (doubleDash) {
        if (value) {
          if (booleanSet.has(key)) value = parseBooleanString(value);
          setArgument(key, value, arg, true);
          continue;
        }
        if (negated) {
          if (negatableSet.has(key)) {
            setArgument(key, false, arg, false);
            continue;
          }
          key = `no-${key}`;
        }
        const next = args[i + 1];
        if (!booleanSet.has(key) && !allBools && next && !/^-/.test(next) && (aliasMap.get(key) ? !aliasIsBoolean(aliasMap, booleanSet, key) : true)) {
          value = next;
          i++;
          setArgument(key, value, arg, true);
          continue;
        }
        if (isBooleanString(next)) {
          value = parseBooleanString(next);
          i++;
          setArgument(key, value, arg, true);
          continue;
        }
        value = stringSet.has(key) ? "" : true;
        setArgument(key, value, arg, true);
        continue;
      }
      const letters = arg.slice(1, -1).split("");
      let broken = false;
      for (const [j, letter] of letters.entries()){
        const next = arg.slice(j + 2);
        if (next === "-") {
          setArgument(letter, next, arg, true);
          continue;
        }
        if (/[A-Za-z]/.test(letter) && /=/.test(next)) {
          setArgument(letter, next.split(/=(.+)/)[1], arg, true);
          broken = true;
          break;
        }
        if (/[A-Za-z]/.test(letter) && /-?\d+(\.\d*)?(e-?\d+)?$/.test(next)) {
          setArgument(letter, next, arg, true);
          broken = true;
          break;
        }
        if (letters[j + 1] && letters[j + 1].match(/\W/)) {
          setArgument(letter, arg.slice(j + 2), arg, true);
          broken = true;
          break;
        }
        setArgument(letter, stringSet.has(letter) ? "" : true, arg, true);
      }
      key = arg.slice(-1);
      if (!broken && key !== "-") {
        const nextArg = args[i + 1];
        if (nextArg && !/^(-|--)[^-]/.test(nextArg) && !booleanSet.has(key) && (aliasMap.get(key) ? !aliasIsBoolean(aliasMap, booleanSet, key) : true)) {
          setArgument(key, nextArg, arg, true);
          i++;
        } else if (nextArg && isBooleanString(nextArg)) {
          const value = parseBooleanString(nextArg);
          setArgument(key, value, arg, true);
          i++;
        } else {
          setArgument(key, stringSet.has(key) ? "" : true, arg, true);
        }
      }
      continue;
    }
    if (unknownFn?.(arg) !== false) {
      argv._.push(stringSet.has("_") || !isNumber(arg) ? arg : Number(arg));
    }
    if (stopEarly) {
      argv._.push(...args.slice(i + 1));
      break;
    }
  }
  for (const [key, value] of Object.entries(defaults)){
    const keys = key.split(".");
    if (!hasNested(argv, keys)) {
      setNested(argv, keys, value);
      aliasMap.get(key)?.forEach((key)=>setNested(argv, key.split("."), value));
    }
  }
  for (const key of booleanSet.keys()){
    const keys = key.split(".");
    if (!hasNested(argv, keys)) {
      const value = collectSet.has(key) ? [] : false;
      setNested(argv, keys, value);
    }
  }
  for (const key of stringSet.keys()){
    const keys = key.split(".");
    if (!hasNested(argv, keys) && collectSet.has(key)) {
      setNested(argv, keys, []);
    }
  }
  if (doubleDash) {
    argv["--"] = [];
    for (const key of notFlags){
      argv["--"].push(key);
    }
  } else {
    for (const key of notFlags){
      argv._.push(key);
    }
  }
  return argv;
}
//# sourceMappingURL=data:application/json;base64,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