// Copyright 2018-2024 the Deno authors. All rights reserved. MIT license.
/**
 * Provides user-friendly {@linkcode serve} on top of Deno's native HTTP server
 * and other utilities for creating HTTP servers and clients.
 *
 * ## File Server
 *
 * A small program for serving local files over HTTP.
 *
 * ```sh
 * deno run --allow-net --allow-read https://deno.land/std/http/file_server.ts
 * > HTTP server listening on http://localhost:4507/
 * ```
 *
 * ## HTTP Status Code and Status Text
 *
 * Helper for processing status code and status text.
 *
 * ## HTTP errors
 *
 * Provides error classes for each HTTP error status code as well as utility
 * functions for handling HTTP errors in a structured way.
 *
 * ## Methods
 *
 * Provides helper functions and types to work with HTTP method strings safely.
 *
 * ## Negotiation
 *
 * A set of functions which can be used to negotiate content types, encodings and
 * languages when responding to requests.
 *
 * > Note: some libraries include accept charset functionality by analyzing the
 * > `Accept-Charset` header. This is a legacy header that
 * > {@link https://developer.mozilla.org/en-US/docs/Web/HTTP/Headers/Accept-Charset | clients omit and servers should ignore}
 * > therefore is not provided.
 *
 * ## Cookie maps
 *
 * An alternative to `cookie.ts` is `cookie_map.ts` which provides `CookieMap`,
 * `SecureCookieMap`, and `mergeHeaders` to manage request and response cookies
 * with the familiar `Map` interface.
 *
 * ## User agent handling
 *
 * The {@linkcode UserAgent} class provides user agent string parsing, allowing
 * a user agent flag to be semantically understood.
 *
 * For example to integrate the user agent provided in the header `User-Agent`
 * in an http request would look like this:
 *
 * ```ts
 * import { UserAgent } from "https://deno.land/std@$STD_VERSION/http/user_agent.ts";
 *
 * Deno.serve((req) => {
 *   const userAgent = new UserAgent(req.headers.get("user-agent") ?? "");
 *   return new Response(`Hello, ${userAgent.browser.name}
 *     on ${userAgent.os.name} ${userAgent.os.version}!`);
 * });
 * ```
 *
 * @module
 */ export * from "./cookie.ts";
export * from "./etag.ts";
export * from "./status.ts";
export * from "./negotiation.ts";
export * from "./server.ts";
export * from "./unstable_signed_cookie.ts";
export * from "./server_sent_event_stream.ts";
export * from "./user_agent.ts";
export * from "./file_server.ts";
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJzb3VyY2VzIjpbImh0dHBzOi8vZGVuby5sYW5kL3N0ZEAwLjIxNy4wL2h0dHAvbW9kLnRzIl0sInNvdXJjZXNDb250ZW50IjpbIi8vIENvcHlyaWdodCAyMDE4LTIwMjQgdGhlIERlbm8gYXV0aG9ycy4gQWxsIHJpZ2h0cyByZXNlcnZlZC4gTUlUIGxpY2Vuc2UuXG4vKipcbiAqIFByb3ZpZGVzIHVzZXItZnJpZW5kbHkge0BsaW5rY29kZSBzZXJ2ZX0gb24gdG9wIG9mIERlbm8ncyBuYXRpdmUgSFRUUCBzZXJ2ZXJcbiAqIGFuZCBvdGhlciB1dGlsaXRpZXMgZm9yIGNyZWF0aW5nIEhUVFAgc2VydmVycyBhbmQgY2xpZW50cy5cbiAqXG4gKiAjIyBGaWxlIFNlcnZlclxuICpcbiAqIEEgc21hbGwgcHJvZ3JhbSBmb3Igc2VydmluZyBsb2NhbCBmaWxlcyBvdmVyIEhUVFAuXG4gKlxuICogYGBgc2hcbiAqIGRlbm8gcnVuIC0tYWxsb3ctbmV0IC0tYWxsb3ctcmVhZCBodHRwczovL2Rlbm8ubGFuZC9zdGQvaHR0cC9maWxlX3NlcnZlci50c1xuICogPiBIVFRQIHNlcnZlciBsaXN0ZW5pbmcgb24gaHR0cDovL2xvY2FsaG9zdDo0NTA3L1xuICogYGBgXG4gKlxuICogIyMgSFRUUCBTdGF0dXMgQ29kZSBhbmQgU3RhdHVzIFRleHRcbiAqXG4gKiBIZWxwZXIgZm9yIHByb2Nlc3Npbmcgc3RhdHVzIGNvZGUgYW5kIHN0YXR1cyB0ZXh0LlxuICpcbiAqICMjIEhUVFAgZXJyb3JzXG4gKlxuICogUHJvdmlkZXMgZXJyb3IgY2xhc3NlcyBmb3IgZWFjaCBIVFRQIGVycm9yIHN0YXR1cyBjb2RlIGFzIHdlbGwgYXMgdXRpbGl0eVxuICogZnVuY3Rpb25zIGZvciBoYW5kbGluZyBIVFRQIGVycm9ycyBpbiBhIHN0cnVjdHVyZWQgd2F5LlxuICpcbiAqICMjIE1ldGhvZHNcbiAqXG4gKiBQcm92aWRlcyBoZWxwZXIgZnVuY3Rpb25zIGFuZCB0eXBlcyB0byB3b3JrIHdpdGggSFRUUCBtZXRob2Qgc3RyaW5ncyBzYWZlbHkuXG4gKlxuICogIyMgTmVnb3RpYXRpb25cbiAqXG4gKiBBIHNldCBvZiBmdW5jdGlvbnMgd2hpY2ggY2FuIGJlIHVzZWQgdG8gbmVnb3RpYXRlIGNvbnRlbnQgdHlwZXMsIGVuY29kaW5ncyBhbmRcbiAqIGxhbmd1YWdlcyB3aGVuIHJlc3BvbmRpbmcgdG8gcmVxdWVzdHMuXG4gKlxuICogPiBOb3RlOiBzb21lIGxpYnJhcmllcyBpbmNsdWRlIGFjY2VwdCBjaGFyc2V0IGZ1bmN0aW9uYWxpdHkgYnkgYW5hbHl6aW5nIHRoZVxuICogPiBgQWNjZXB0LUNoYXJzZXRgIGhlYWRlci4gVGhpcyBpcyBhIGxlZ2FjeSBoZWFkZXIgdGhhdFxuICogPiB7QGxpbmsgaHR0cHM6Ly9kZXZlbG9wZXIubW96aWxsYS5vcmcvZW4tVVMvZG9jcy9XZWIvSFRUUC9IZWFkZXJzL0FjY2VwdC1DaGFyc2V0IHwgY2xpZW50cyBvbWl0IGFuZCBzZXJ2ZXJzIHNob3VsZCBpZ25vcmV9XG4gKiA+IHRoZXJlZm9yZSBpcyBub3QgcHJvdmlkZWQuXG4gKlxuICogIyMgQ29va2llIG1hcHNcbiAqXG4gKiBBbiBhbHRlcm5hdGl2ZSB0byBgY29va2llLnRzYCBpcyBgY29va2llX21hcC50c2Agd2hpY2ggcHJvdmlkZXMgYENvb2tpZU1hcGAsXG4gKiBgU2VjdXJlQ29va2llTWFwYCwgYW5kIGBtZXJnZUhlYWRlcnNgIHRvIG1hbmFnZSByZXF1ZXN0IGFuZCByZXNwb25zZSBjb29raWVzXG4gKiB3aXRoIHRoZSBmYW1pbGlhciBgTWFwYCBpbnRlcmZhY2UuXG4gKlxuICogIyMgVXNlciBhZ2VudCBoYW5kbGluZ1xuICpcbiAqIFRoZSB7QGxpbmtjb2RlIFVzZXJBZ2VudH0gY2xhc3MgcHJvdmlkZXMgdXNlciBhZ2VudCBzdHJpbmcgcGFyc2luZywgYWxsb3dpbmdcbiAqIGEgdXNlciBhZ2VudCBmbGFnIHRvIGJlIHNlbWFudGljYWxseSB1bmRlcnN0b29kLlxuICpcbiAqIEZvciBleGFtcGxlIHRvIGludGVncmF0ZSB0aGUgdXNlciBhZ2VudCBwcm92aWRlZCBpbiB0aGUgaGVhZGVyIGBVc2VyLUFnZW50YFxuICogaW4gYW4gaHR0cCByZXF1ZXN0IHdvdWxkIGxvb2sgbGlrZSB0aGlzOlxuICpcbiAqIGBgYHRzXG4gKiBpbXBvcnQgeyBVc2VyQWdlbnQgfSBmcm9tIFwiaHR0cHM6Ly9kZW5vLmxhbmQvc3RkQCRTVERfVkVSU0lPTi9odHRwL3VzZXJfYWdlbnQudHNcIjtcbiAqXG4gKiBEZW5vLnNlcnZlKChyZXEpID0+IHtcbiAqICAgY29uc3QgdXNlckFnZW50ID0gbmV3IFVzZXJBZ2VudChyZXEuaGVhZGVycy5nZXQoXCJ1c2VyLWFnZW50XCIpID8/IFwiXCIpO1xuICogICByZXR1cm4gbmV3IFJlc3BvbnNlKGBIZWxsbywgJHt1c2VyQWdlbnQuYnJvd3Nlci5uYW1lfVxuICogICAgIG9uICR7dXNlckFnZW50Lm9zLm5hbWV9ICR7dXNlckFnZW50Lm9zLnZlcnNpb259IWApO1xuICogfSk7XG4gKiBgYGBcbiAqXG4gKiBAbW9kdWxlXG4gKi9cblxuZXhwb3J0ICogZnJvbSBcIi4vY29va2llLnRzXCI7XG5leHBvcnQgKiBmcm9tIFwiLi9ldGFnLnRzXCI7XG5leHBvcnQgKiBmcm9tIFwiLi9zdGF0dXMudHNcIjtcbmV4cG9ydCAqIGZyb20gXCIuL25lZ290aWF0aW9uLnRzXCI7XG5leHBvcnQgKiBmcm9tIFwiLi9zZXJ2ZXIudHNcIjtcbmV4cG9ydCAqIGZyb20gXCIuL3Vuc3RhYmxlX3NpZ25lZF9jb29raWUudHNcIjtcbmV4cG9ydCAqIGZyb20gXCIuL3NlcnZlcl9zZW50X2V2ZW50X3N0cmVhbS50c1wiO1xuZXhwb3J0ICogZnJvbSBcIi4vdXNlcl9hZ2VudC50c1wiO1xuZXhwb3J0ICogZnJvbSBcIi4vZmlsZV9zZXJ2ZXIudHNcIjtcbiJdLCJuYW1lcyI6W10sIm1hcHBpbmdzIjoiQUFBQSwwRUFBMEU7QUFDMUU7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Q0E2REMsR0FFRCxjQUFjLGNBQWM7QUFDNUIsY0FBYyxZQUFZO0FBQzFCLGNBQWMsY0FBYztBQUM1QixjQUFjLG1CQUFtQjtBQUNqQyxjQUFjLGNBQWM7QUFDNUIsY0FBYyw4QkFBOEI7QUFDNUMsY0FBYyxnQ0FBZ0M7QUFDOUMsY0FBYyxrQkFBa0I7QUFDaEMsY0FBYyxtQkFBbUIifQ==