// Copyright 2018-2024 the Deno authors. All rights reserved. MIT license.
/**
 * Merge multiple streams into a single one, not taking order into account.
 * If a stream ends before other ones, the other will continue adding data,
 * and the finished one will not add any more data.
 *
 * @example
 * ```ts
 * import { mergeReadableStreams } from "https://deno.land/std@$STD_VERSION/streams/merge_readable_streams.ts";
 *
 * const stream1 = ReadableStream.from(["1", "2", "3"]);
 * const stream2 = ReadableStream.from(["a", "b", "c"]);
 *
 * // ["2", "c", "a", "b", "3", "1"]
 * await Array.fromAsync(mergeReadableStreams(stream1, stream2));
 * ```
 */ export function mergeReadableStreams(...streams) {
  const resolvePromises = streams.map(()=>Promise.withResolvers());
  return new ReadableStream({
    start (controller) {
      let mustClose = false;
      Promise.all(resolvePromises.map(({ promise })=>promise)).then(()=>{
        controller.close();
      }).catch((error)=>{
        mustClose = true;
        controller.error(error);
      });
      for (const [index, stream] of streams.entries()){
        (async ()=>{
          try {
            for await (const data of stream){
              if (mustClose) {
                break;
              }
              controller.enqueue(data);
            }
            resolvePromises[index].resolve();
          } catch (error) {
            resolvePromises[index].reject(error);
          }
        })();
      }
    }
  });
}
//# sourceMappingURL=data:application/json;base64,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