// Copyright 2018-2024 the Deno authors. All rights reserved. MIT license.
/**
 * Provides functions for dealing with and matching ETags, including
 * {@linkcode calculate} to calculate an etag for a given entity,
 * {@linkcode ifMatch} for validating if an ETag matches against a `If-Match`
 * header and {@linkcode ifNoneMatch} for validating an Etag against an
 * `If-None-Match` header.
 *
 * See further information on the `ETag` header on
 * {@link https://developer.mozilla.org/en-US/docs/Web/HTTP/Headers/ETag | MDN}.
 *
 * @module
 */ import { encodeBase64 as base64Encode } from "../encoding/base64.ts";
const encoder = new TextEncoder();
const DEFAULT_ALGORITHM = "SHA-256";
function isFileInfo(value) {
  return Boolean(value && typeof value === "object" && "mtime" in value && "size" in value);
}
async function calcEntity(entity, { algorithm = DEFAULT_ALGORITHM }) {
  // a short circuit for zero length entities
  if (entity.length === 0) {
    return `0-47DEQpj8HBSa+/TImW+5JCeuQeR`;
  }
  if (typeof entity === "string") {
    entity = encoder.encode(entity);
  }
  const hash = base64Encode(await crypto.subtle.digest(algorithm, entity)).substring(0, 27);
  return `${entity.length.toString(16)}-${hash}`;
}
async function calcFileInfo(fileInfo, { algorithm = DEFAULT_ALGORITHM }) {
  if (fileInfo.mtime) {
    const hash = base64Encode(await crypto.subtle.digest(algorithm, encoder.encode(fileInfo.mtime.toJSON()))).substring(0, 27);
    return `${fileInfo.size.toString(16)}-${hash}`;
  }
}
/**
 * Calculate an ETag for an entity. When the entity is a specific set of data
 * it will be fingerprinted as a "strong" tag, otherwise if it is just file
 * information, it will be calculated as a weak tag.
 *
 * ```ts
 * import { calculate } from "https://deno.land/std@$STD_VERSION/http/etag.ts";
 * import { assert } from "https://deno.land/std@$STD_VERSION/assert/assert.ts"
 *
 * const body = "hello deno!";
 *
 * const etag = await calculate(body);
 * assert(etag);
 *
 * const res = new Response(body, { headers: { etag } });
 * ```
 */ export async function calculate(entity, options = {}) {
  const weak = options.weak ?? isFileInfo(entity);
  const tag = await (isFileInfo(entity) ? calcFileInfo(entity, options) : calcEntity(entity, options));
  return tag ? weak ? `W/"${tag}"` : `"${tag}"` : undefined;
}
/** A helper function that takes the value from the `If-Match` header and a
 * calculated etag for the target. By using strong comparison, return `true` if
 * the values match, otherwise `false`.
 *
 * See MDN's [`If-Match`](https://developer.mozilla.org/en-US/docs/Web/HTTP/Headers/If-Match)
 * article for more information on how to use this function.
 *
 * ```ts
 * import {
 *   calculate,
 *   ifMatch,
 * } from "https://deno.land/std@$STD_VERSION/http/etag.ts";
 * import { assert } from "https://deno.land/std@$STD_VERSION/assert/assert.ts"
 *
 * const body = "hello deno!";
 *
 * Deno.serve(async (req) => {
 *   const ifMatchValue = req.headers.get("if-match");
 *   const etag = await calculate(body);
 *   assert(etag);
 *   if (!ifMatchValue || ifMatch(ifMatchValue, etag)) {
 *     return new Response(body, { status: 200, headers: { etag } });
 *   } else {
 *     return new Response(null, { status: 412, statusText: "Precondition Failed"});
 *   }
 * });
 * ```
 */ export function ifMatch(value, etag) {
  // Weak tags cannot be matched and return false.
  if (!value || !etag || etag.startsWith("W/")) {
    return false;
  }
  if (value.trim() === "*") {
    return true;
  }
  const tags = value.split(/\s*,\s*/);
  return tags.includes(etag);
}
/** A helper function that takes the value from the `If-None-Match` header and
 * a calculated etag for the target entity and returns `false` if the etag for
 * the entity matches the supplied value, otherwise `true`.
 *
 * See MDN's [`If-None-Match`](https://developer.mozilla.org/en-US/docs/Web/HTTP/Headers/If-None-Match)
 * article for more information on how to use this function.
 *
 * ```ts
 * import {
 *   calculate,
 *   ifNoneMatch,
 * } from "https://deno.land/std@$STD_VERSION/http/etag.ts";
 * import { assert } from "https://deno.land/std@$STD_VERSION/assert/assert.ts"
 *
 * const body = "hello deno!";
 *
 * Deno.serve(async (req) => {
 *   const ifNoneMatchValue = req.headers.get("if-none-match");
 *   const etag = await calculate(body);
 *   assert(etag);
 *   if (!ifNoneMatch(ifNoneMatchValue, etag)) {
 *     return new Response(null, { status: 304, headers: { etag } });
 *   } else {
 *     return new Response(body, { status: 200, headers: { etag } });
 *   }
 * });
 * ```
 */ export function ifNoneMatch(value, etag) {
  if (!value || !etag) {
    return true;
  }
  if (value.trim() === "*") {
    return false;
  }
  etag = etag.startsWith("W/") ? etag.slice(2) : etag;
  const tags = value.split(/\s*,\s*/).map((tag)=>tag.startsWith("W/") ? tag.slice(2) : tag);
  return !tags.includes(etag);
}
//# sourceMappingURL=data:application/json;base64,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