// Copyright 2018-2024 the Deno authors. All rights reserved. MIT license.
// This module is browser compatible.
/** Options for {@linkcode TextLineStream}. */ /**
 * Transform a stream into a stream where each chunk is divided by a newline,
 * be it `\n` or `\r\n`. `\r` can be enabled via the `allowCR` option.
 *
 * @example
 * ```ts
 * import { TextLineStream } from "https://deno.land/std@$STD_VERSION/streams/text_line_stream.ts";
 *
 * const res = await fetch("https://example.com");
 * const lines = res.body!
 *   .pipeThrough(new TextDecoderStream())
 *   .pipeThrough(new TextLineStream());
 * ```
 */ export class TextLineStream extends TransformStream {
  #currentLine = "";
  /** Constructs a new instance. */ constructor(options = {
    allowCR: false
  }){
    super({
      transform: (chars, controller)=>{
        chars = this.#currentLine + chars;
        while(true){
          const lfIndex = chars.indexOf("\n");
          const crIndex = options.allowCR ? chars.indexOf("\r") : -1;
          if (crIndex !== -1 && crIndex !== chars.length - 1 && (lfIndex === -1 || lfIndex - 1 > crIndex)) {
            controller.enqueue(chars.slice(0, crIndex));
            chars = chars.slice(crIndex + 1);
            continue;
          }
          if (lfIndex === -1) break;
          const endIndex = chars[lfIndex - 1] === "\r" ? lfIndex - 1 : lfIndex;
          controller.enqueue(chars.slice(0, endIndex));
          chars = chars.slice(lfIndex + 1);
        }
        this.#currentLine = chars;
      },
      flush: (controller)=>{
        if (this.#currentLine === "") return;
        const currentLine = options.allowCR && this.#currentLine.endsWith("\r") ? this.#currentLine.slice(0, -1) : this.#currentLine;
        controller.enqueue(currentLine);
      }
    });
  }
}
//# sourceMappingURL=data:application/json;base64,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