// Copyright 2018-2024 the Deno authors. All rights reserved. MIT license.
// This module is browser compatible.
/**
 * Applies the given transformer to all values in the given record and returns a
 * new record containing the resulting keys associated to the last value that
 * produced them.
 *
 * @example
 * ```ts
 * import { mapValues } from "https://deno.land/std@$STD_VERSION/collections/map_values.ts";
 * import { assertEquals } from "https://deno.land/std@$STD_VERSION/assert/assert_equals.ts";
 *
 * const usersById = {
 *   "a5ec": { name: "Mischa" },
 *   "de4f": { name: "Kim" },
 * };
 * const namesById = mapValues(usersById, (it) => it.name);
 *
 * assertEquals(
 *   namesById,
 *   {
 *     "a5ec": "Mischa",
 *     "de4f": "Kim",
 *   },
 * );
 * ```
 */ export function mapValues(record, transformer) {
  // deno-lint-ignore no-explicit-any
  const ret = {};
  const entries = Object.entries(record);
  for (const [key, value] of entries){
    const mappedValue = transformer(value, key);
    ret[key] = mappedValue;
  }
  return ret;
}
//# sourceMappingURL=data:application/json;base64,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