// Copyright 2018-2024 the Deno authors. All rights reserved. MIT license.
// This module is browser compatible.
/**
 * Utilities for
 * {@link https://datatracker.ietf.org/doc/html/draft-msporny-base58-03 | base58}
 * encoding and decoding.
 *
 * This module is browser compatible.
 *
 * @module
 */ import { validateBinaryLike } from "./_util.ts";
// deno-fmt-ignore
const mapBase58 = {
  "1": 0,
  "2": 1,
  "3": 2,
  "4": 3,
  "5": 4,
  "6": 5,
  "7": 6,
  "8": 7,
  "9": 8,
  A: 9,
  B: 10,
  C: 11,
  D: 12,
  E: 13,
  F: 14,
  G: 15,
  H: 16,
  J: 17,
  K: 18,
  L: 19,
  M: 20,
  N: 21,
  P: 22,
  Q: 23,
  R: 24,
  S: 25,
  T: 26,
  U: 27,
  V: 28,
  W: 29,
  X: 30,
  Y: 31,
  Z: 32,
  a: 33,
  b: 34,
  c: 35,
  d: 36,
  e: 37,
  f: 38,
  g: 39,
  h: 40,
  i: 41,
  j: 42,
  k: 43,
  m: 44,
  n: 45,
  o: 46,
  p: 47,
  q: 48,
  r: 49,
  s: 50,
  t: 51,
  u: 52,
  v: 53,
  w: 54,
  x: 55,
  y: 56,
  z: 57
};
const base58alphabet = "123456789ABCDEFGHJKLMNPQRSTUVWXYZabcdefghijkmnopqrstuvwxyz".split("");
/**
 * Converts data to a base58-encoded string.
 *
 * @see {@link https://datatracker.ietf.org/doc/html/draft-msporny-base58-03#section-3}
 *
 * @example
 * ```ts
 * import { encodeBase58 } from "https://deno.land/std@$STD_VERSION/encoding/base58.ts";
 *
 * encodeBase58("Hello World!"); // "2NEpo7TZRRrLZSi2U"
 * ```
 */ export function encodeBase58(data) {
  const uint8tData = validateBinaryLike(data);
  let length = 0;
  let zeroes = 0;
  // Counting leading zeroes
  let index = 0;
  while(uint8tData[index] === 0){
    zeroes++;
    index++;
  }
  const notZeroUint8Data = uint8tData.slice(index);
  const size = Math.round(uint8tData.length * 138 / 100 + 1);
  const b58Encoding = [];
  notZeroUint8Data.forEach((byte)=>{
    let i = 0;
    let carry = byte;
    for(let reverse_iterator = size - 1; (carry > 0 || i < length) && reverse_iterator !== -1; reverse_iterator--, i++){
      carry += (b58Encoding[reverse_iterator] || 0) * 256;
      b58Encoding[reverse_iterator] = Math.round(carry % 58);
      carry = Math.floor(carry / 58);
    }
    length = i;
  });
  const strResult = Array.from({
    length: b58Encoding.length + zeroes
  });
  if (zeroes > 0) {
    strResult.fill("1", 0, zeroes);
  }
  b58Encoding.forEach((byteValue)=>strResult.push(base58alphabet[byteValue]));
  return strResult.join("");
}
/**
 * Decodes a base58-encoded string.
 *
 * @see {@link https://datatracker.ietf.org/doc/html/draft-msporny-base58-03#section-4}
 *
 * @example
 * ```ts
 * import { decodeBase58 } from "https://deno.land/std@$STD_VERSION/encoding/base58.ts";
 *
 * decodeBase58("2NEpo7TZRRrLZSi2U"); // Uint8Array(12) [ 72, 101, 108, 108, 111, 32,  87, 111, 114, 108, 100, 33 ]
 * ```
 */ export function decodeBase58(b58) {
  const splitInput = b58.trim().split("");
  let length = 0;
  let ones = 0;
  // Counting leading ones
  let index = 0;
  while(splitInput[index] === "1"){
    ones++;
    index++;
  }
  const notZeroData = splitInput.slice(index);
  const size = Math.round(b58.length * 733 / 1000 + 1);
  const output = [];
  notZeroData.forEach((char, idx)=>{
    let carry = mapBase58[char];
    let i = 0;
    if (carry === undefined) {
      throw new Error(`Invalid base58 char at index ${idx} with value ${char}`);
    }
    for(let reverse_iterator = size - 1; (carry > 0 || i < length) && reverse_iterator !== -1; reverse_iterator--, i++){
      carry += 58 * (output[reverse_iterator] || 0);
      output[reverse_iterator] = Math.round(carry % 256);
      carry = Math.floor(carry / 256);
    }
    length = i;
  });
  const validOutput = output.filter((item)=>item !== undefined);
  if (ones > 0) {
    const onesResult = Array.from({
      length: ones
    }).fill(0, 0, ones);
    return new Uint8Array([
      ...onesResult,
      ...validOutput
    ]);
  }
  return new Uint8Array(validOutput);
}
//# sourceMappingURL=data:application/json;base64,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